require 'test_helper'

class OrderTest < ActiveSupport::TestCase

  def setup
    @alex = users(:alex)
    @article_one = articles(:article_one)
    @menu_order_two = menu_orders(:menu_order_two)
  end

  test "value paid is defaulted to false" do
    @order = Order.new :ordered_at => Time.now, :price => 1
    assert @order.save

    assert_equal false, @order.paid
  end

  test "ordered_at is not null" do
    @order = Order.new :price => 1
    assert !@order.save

    @order.ordered_at = Time.now
    assert @order.save
  end

  test "ordered_at is a date" do
    @order = Order.new :ordered_at => "NoTime", :price => 1
    assert !@order.save

    @order.ordered_at = Time.now
    assert @order.save
  end

  test "order belongs to a user" do
    @order = Order.new :ordered_at => Time.now, :price => 1
    @order.user = @alex
    assert_equal @alex, @order.user
  end

  test "order belongs to an article" do
    @order = Order.new :ordered_at => Time.now, :price => 1
    @order.article = @article_one
    assert_equal @article_one, @order.article
  end

  test "order needs a price" do
    @order = Order.new :ordered_at => Time.now
    assert !@order.save

    @order.price = 10
    assert @order.save
  end

  test "price is a number" do
    @order = Order.new :ordered_at => Time.now, :price => "notANumber"
    assert !@order.save
  end

  test "price is not smaller than zero" do
    @order = Order.new :ordered_at => Time.now, :price => -1
    assert !@order.save

    @order.price = 0
    assert @order.save
  end

  test "price is not greater than or equal to one million" do
    @order = Order.new :ordered_at => Time.now, :price => 1000000
    assert !@order.save

    @order = Order.new :ordered_at => Time.now, :price => 999999
    assert @order.save
  end

  test "amount has only two digits to the right of the decimal point" do
    @order = Order.new :ordered_at => Time.now, :price => 10.005
    assert @order.save
    #this should round the amount to 10.01

    @order = Order.find_by_price(10.01)
    assert_not_nil @order
  end

  #testing polymorphic association
  test "main and sub order association is working" do
    @order = Order.new :ordered_at => Time.now, :price => 10
    @order.sub_order = @menu_order_two
    @menu_order_two.main_order = @order
    @mot = MenuOrder.find_by_id @menu_order_two.id
    
    assert_equal @menu_order_two, @order.sub_order
    assert_equal @menu_order_two.main_order, @order
  end
end
