require 'test_helper'

class UsersControllerTest < ActionController::TestCase
  setup :activate_authlogic

  def setup
    login_as(:alex)

    # load users from fixtures
    @alex = users(:alex)
    @david = users(:david)

    # load groups from fixtures
    @teachers = groups(:teachers)
    @pupils = groups(:pupils)
  end

  test "user is set for edit in profile" do
    get :edit

    # User should be set to be edited
    assert_not_nil assigns(:user)

    # User Object should be the same as logged in User
    assert_equal assigns(:user), assigns(:current_user)
    assert_template "edit"
  end

  test "different user is set for edit " do
    # We are logged in as alex, therefore david is a different user
    get :edit, :id => @david.id

    # User should be set to be edited
    assert_not_nil assigns(:user)
    assert_equal assigns(:user), @david

    assert_response :success
    assert_template "edit_all_features"
  end

  test "user account data is updated" do
    post :update, :user=>{:email => "alex@newmail.org", :password => "moresecure", :password_confirmation => "moresecure", :login => '10', :forename => 'AlexChanged', :surname => 'NewSmith'}

    # user may update his email and password
    assert_equal assigns(:current_user).email, "alex@newmail.org"
    assert_equal assigns(:current_user).password, "moresecure"

    # check that user can't update login, forename, surname and associated groups
    assert_equal @alex.login, assigns(:current_user).login
    assert_equal @alex.forename, assigns(:current_user).forename
    assert_equal @alex.surname, assigns(:current_user).surname
    assert_equal @alex.group_ids, assigns(:current_user).group_ids
    assert_redirected_to edit_account_url
  end

  test "user data is updated correctly" do
    post :update_all, :id => @david.id, :user=>{:email => "david@newmail.org", :password => "moresecure", :password_confirmation => "moresecure", :forename => 'NewDavid', :surname => 'NewThompson', :group_ids =>[@teachers.id, @pupils.id]}

    # All attributes should be updated!
    @user = User.find @david.id
    assert_equal @user.email, "david@newmail.org"
    assert_equal assigns(:user).password, "moresecure"
    assert_equal @user.forename, "NewDavid"
    assert_equal @user.surname, "NewThompson"
    assert_equal @user.groups.size, 2

    # direct us to the users list to select the next user to edit
    assert_redirected_to users_url
  end

  test "users are set in list" do
    get :index

    assert_response :success
    assert_not_nil assigns(:users)
  end

  test "user is deleted" do
    assert_difference('User.count', -1) do  
      delete :destroy, {:id => @david}
    end
    assert_redirected_to users_url
  end

  test "that user cant delete himself" do
    assert_no_difference('User.count') do
      delete :destroy, {:id => @alex.id}
    end
    assert_equal flash[:error], "Sie können sich nicht selbst löschen!"
    assert_redirected_to users_url
  end

  test "that user is added" do
    assert_difference('User.count', 1) do
      post :create, :user => { :login => '10000', :forename => 'Martha', :surname => 'Musterwoman', :password => 'secure', :password_confirmation => 'secure'}
    end

    assert_redirected_to new_user_url
  end

  test "failure to validate new user" do
    assert_no_difference('User.count') do
      # No login given and wron password_confirmation therefore this should not be saved
      post :create, :user => { :login => '', :forename => 'Martha', :surname => 'Musterwoman', :password => 'secure', :password_confirmation => 'insecure'}
    end

    assert_response :success
  end
end
