require 'test_helper'

class MealtimesControllerTest < ActionController::TestCase
  setup :activate_authlogic

  def setup
    login_as(:alex)
    @early_mealtime = mealtimes(:early_mealtime)
  end

  test "new mealtime is set" do
    get :new

    assert_not_nil assigns(:mealtime)
    assert_template :new
  end

  test "mealtime is created and saved with parameters" do
    assert_difference 'Mealtime.count', 1 do
      post :create, :mealtime => {:name => "new Mealtime", :rank => 100, :start_time => '14:00:00', :end_time => '15:00:00', :order_limit => 100}
    end

    assert_equal flash[:notice], "Essenszeit #{assigns(:mealtime).name} wurde erfolgreich angelegt!"
    assert_redirected_to mealtimes_url
  end

  test "mealtime is not created without parameters" do
    assert_no_difference 'Mealtime.count' do
      post :create, :mealtime => {:name => "", :rank => '', :start_time => '', :end_time => '', :order_limit => ''}
    end

    # success does not mean the Mealtime was saved but as the template is rendered HTML Code is 200 => success
    assert_response :success
    assert_template :new
  end

  test "index loads mealtimes" do
    assert_no_difference 'Mealtime.count' do
      get :index
    end

    assert_not_nil assigns(:mealtimes)
    assert_template :index
    assert_response :success
  end

  test "index loads mealtimes ordered by rank" do
    @mealtime = Mealtime.new :name => "new Mealtime", :rank => 1000, :start_time => '14:00:00', :end_time => '15:00:00', :order_limit => 100
    @mealtime.save
    @mealtime2 = Mealtime.new :name => "new Mealtime2", :rank => 999, :start_time => '15:00:00', :end_time => '16:00:00', :order_limit => 100
    @mealtime2.save

    get :index

    assert_equal assigns(:mealtimes).last, @mealtime
  end

  test "mealtime is updated with new values" do
    assert_no_difference 'Mealtime.count' do
      post :update, :id => @early_mealtime.id, :mealtime => {:name => "NameUpdated", :rank => 106, :start_time => '09:00:00', :end_time => '10:00:00', :order_limit => 101}
    end

    # get recently changed mealtime and test values
    @early_mealtime = Mealtime.find @early_mealtime.id
    assert_equal "NameUpdated", @early_mealtime.name
    assert_equal 106, @early_mealtime.rank
    assert_equal '2000-01-01 09:00:00'.to_time, @early_mealtime.start_time
    assert_equal '2000-01-01 10:00:00'.to_time, @early_mealtime.end_time
    assert_equal 101, @early_mealtime.order_limit
    assert_equal "Änderungen erfolgreich gespeichert", flash[:notice]
    assert_redirected_to mealtimes_url
  end

  test "mealtime update not successfull with missing parameters" do
    @mealtime = Mealtime.new :name => "Mealtime", :rank => 1000, :start_time => '09:00:00', :end_time => '10:00:00', :order_limit => 100
    assert @mealtime.save
    post :update, :id => @mealtime.id, :mealtime => {:name => "", :rank => '', :start_time => '', :end_time => '', :order_limit => ''}

    #check if update did not affect the mealtime
    @mealtime = Mealtime.find @mealtime.id
    assert_equal @mealtime.name, "Mealtime"
    assert_equal @mealtime.rank, 1000
    assert_equal @mealtime.start_time, '2000-01-01 09:00:00'.to_time
    assert_equal @mealtime.end_time, '2000-01-01 10:00:00'.to_time
    assert_equal @mealtime.order_limit, 100

    # Edit form should be shown to the user
    assert_template :edit
  end

  test "edit form is initialized with mealtime object" do
    get :edit, :id => @early_mealtime.id

    assert_equal assigns(:mealtime), @early_mealtime
    assert_template :edit
    assert_response :success
  end

  test "edit form not shown when id is not known" do
    get :edit

    assert_equal "Eine Essenszeit mit dieser ID gibt es nicht!", flash[:error]
    assert_redirected_to mealtimes_url
  end

  test "mealtime is deleted by id" do
    @mealtime = Mealtime.new :name => "Mealtime", :rank => 500, :start_time => '09:00:00', :end_time => '10:00:00', :order_limit => 100
    @mealtime.save

    assert_difference 'Mealtime.count', -1 do
      delete :destroy, :id => @mealtime.id
    end

    assert_equal flash[:notice], "Essenszeit #{@mealtime.name} erfolgreich gelöscht!"
    assert_redirected_to mealtimes_url
  end
end
