require 'test_helper'

class MealsControllerTest < ActionController::TestCase
  setup :activate_authlogic

  def setup
    login_as(:alex)
    @spaghetti = meals(:spaghetti)
    @main_menu = categories(:main_menu)
    @snack = categories(:snack)
  end

  test "new meal is set" do
    get :new

    assert_not_nil assigns(:meal)
    assert_template :new
  end

  test "meal is created and saved with parameters" do
    assert_difference 'Meal.count', 1 do
      post :create, :meal => {:description => "new Meal", :number => 100, :category_id => @main_menu.id}
    end

    assert_equal flash[:notice], "Speise #{assigns(:meal).description} wurde erfolgreich angelegt!"
    assert_redirected_to meals_url
  end

  test "meal is not created without parameters" do
    assert_no_difference 'Meal.count' do
      post :create, :meal => {:description => "", :number => '', :category_id => ''}
    end

    # success does not mean the Meal was saved but as the template is rendered HTML Code is 200 => success
    assert_response :success
    assert_template :new
  end

  test "index loads meals" do
    assert_no_difference 'Meal.count' do
      get :index
    end

    assert_not_nil assigns(:meals)
    assert_template :index
    assert_response :success
  end

  test "index loads meals ordered by number" do
    @meal = Meal.new :description => "new Meal", :number => 1000, :category_id => @snack.id
    @meal.save
    @meal2 = Meal.new :description => "new Meal2", :number => 999, :category_id => @snack.id
    @meal2.save

    get :index

    assert_equal assigns(:meals).last, @meal
  end

  test "meal is updated with new values" do
    assert_no_difference 'Meal.count' do
      post :update, :id => @spaghetti.id, :meal => {:description => "descriptionUpdated", :number => 20, :category_id => @snack.id}
    end

    # get recently changed meal and test values
    @spaghetti = Meal.find @spaghetti.id
    assert_equal "descriptionUpdated", @spaghetti.description
    assert_equal 20, @spaghetti.number
    assert_equal @snack.id, @spaghetti.category_id
    assert_equal "Änderungen erfolgreich gespeichert", flash[:notice]
    assert_redirected_to meals_url
  end

  test "meal update not successfull with missing parameters" do
    @meal = Meal.new :description => "Meal", :number => 1000, :category_id => @main_menu.id
    assert @meal.save
    post :update, :id => @meal.id, :meal => {:description => "", :number => '', :category_id => ''}

    #check if update did not affect the meal
    @meal = Meal.find @meal.id
    assert_equal @meal.description, "Meal"
    assert_equal @meal.number, 1000
    assert_equal @meal.category_id, @main_menu.id

    # Edit form should be shown to the user
    assert_template :edit
  end

  test "edit form is initialized with meal object" do
    get :edit, :id => @spaghetti.id

    assert_equal assigns(:meal), @spaghetti
    assert_template :edit
    assert_response :success
  end

  test "edit form not shown when id is not known" do
    get :edit

    assert_equal "Eine Speise mit dieser ID gibt es nicht!", flash[:error]
    assert_redirected_to meals_url
  end

  test "meal is deleted by id" do
    @meal = Meal.new :description => "Meal", :number => 500, :category_id => @main_menu.id
    @meal.save

    assert_difference 'Meal.count', -1 do
      delete :destroy, :id => @meal.id
    end

    assert_equal flash[:notice], "Speise #{@meal.description} erfolgreich gelöscht!"
    assert_redirected_to meals_url
  end
end
