require 'test_helper'

class GroupsControllerTest < ActionController::TestCase
  setup :activate_authlogic

  def setup
    login_as(:alex)
    @alex = users(:alex)
    @teachers = groups(:teachers)
    @pupils = groups(:pupils)
  end

  test "actions need login" do
    logout
    get :new
    assert_redirected_to login_url
    
    post :create
    assert_redirected_to login_url

    post :update
    assert_redirected_to login_url

    get :edit
    assert_redirected_to login_url

    delete :destroy
    assert_redirected_to login_url
  end

  test "new group is set" do
    get :new

    assert_not_nil assigns(:group)
    assert_template :new
  end

  test "group is created and saved with parameters" do
    assert_difference 'Group.count', 1 do
      post :create, :group=>{:name => "newGroup", :rank => 200}
    end

    assert_equal flash[:notice], "Gruppe #{assigns(:group).name} wurde erfolgreich angelegt!"
    assert_redirected_to new_group_url
  end

  test "group is not created without parameters" do
    assert_no_difference 'Group.count' do
      post :create, :group=>{:name => "", :rank => ''}
    end

    assert_response :success
    assert_template :new
  end

  test "index loads groups" do
    assert_no_difference 'Group.count' do
      get :index
    end

    assert_not_nil assigns(:groups)
    assert_template :index
    assert_response :success
  end

  test "index loads groups ordered by rank" do
    @group = Group.new :name => "orderGroup", :rank => 1000
    @group.save
    @group2 = Group.new :name => "orderGroup2", :rank => 999
    @group2.save

    get :index

    assert_equal assigns(:groups).last, @group
  end

  test "group is updated with new values" do
    assert_no_difference 'Group.count' do
      post :update, :id => @teachers.id, :group => {:name => "teachersNew", :rank => 201}
    end

    # get recently changed group and test values
    @group = Group.find @teachers.id
    assert_equal @group.name, "teachersNew"
    assert_equal @group.rank, 201
    assert_equal flash[:notice], "Änderungen erfolgreich gespeichert"
    assert_redirected_to groups_url
  end

  test "group update not successfull without all parameters" do
    post :update, :id => @pupils.id, :group => {:name => "", :rank => ''}

    #check if update did not affect Group @pupils
    @group = Group.find @pupils.id
    assert_equal @group.name, "pupils"
    assert_equal @group.rank, 2

    # Edit form should be shown to the user
    assert_template :edit
  end

  test "edit form is initialized with group object" do
    get :edit, :id => @teachers.id

    assert_equal assigns(:group), @teachers
    assert_template :edit
    assert_response :success
  end

  test "edit form not shown when id is not known" do
    get :edit

    assert_equal "Eine Gruppe mit dieser ID gibt es nicht!", flash[:error]
    assert_redirected_to groups_url
  end

  test "group is deleted by id" do
    @group = Group.new :name => "deleteGruppe", :rank => 202
    @group.save

    assert_difference 'Group.count', -1 do
      delete :destroy, :id => @group.id
    end

    assert_equal flash[:notice], "Gruppe #{@group.name} erfolgreich gelöscht!"
    assert_redirected_to groups_url
  end
end
