require 'test_helper'

class CategoriesControllerTest < ActionController::TestCase
  setup :activate_authlogic

  def setup
    login_as(:alex)
    @main_menu = categories(:main_menu)
  end

  test "new category is set" do
    get :new

    assert_not_nil assigns(:category)
    assert_template :new
  end

  test "category is created and saved with parameters" do
    assert_difference 'Category.count', 1 do
      post :create, :category=>{:name => "newCat", :rank => 100, :orderable => true, :preorderable => true}
    end

    assert_equal flash[:notice], "Kategorie #{assigns(:category).name} wurde erfolgreich angelegt!"
    assert_redirected_to categories_url
  end

  test "category is not created without parameters" do
    assert_no_difference 'Category.count' do
      post :create, :category=>{:name => "", :rank => '', :orderable => '', :preorderable => ''}
    end

    # success does not mean the Category was saved but as the template is rendered HTML Code is 200 => success
    assert_response :success
    assert_template :new
  end

  test "index loads categories" do
    assert_no_difference 'Category.count' do
      get :index
    end

    assert_not_nil assigns(:categories)
    assert_template :index
    assert_response :success
  end

  test "index loads categories ordered by rank" do
    @category = Category.new :name => "newCat", :rank => 1000, :orderable => true, :preorderable => true
    @category.save
    @category2 = Category.new :name => "newCat2", :rank => 999, :orderable => true, :preorderable => true
    @category2.save

    get :index

    assert_equal assigns(:categories).last, @category
  end

  test "category is updated with new values" do
    assert_no_difference 'Category.count' do
      post :update, :id => @main_menu.id, :category => {:name => "newCategory_update", :rank => 106, :orderable => false, :preorderable => false}
    end

    # get recently changed category and test values
    @main_menu = Category.find @main_menu.id
    assert_equal "newCategory_update", @main_menu.name
    assert_equal 106, @main_menu.rank
    assert_equal false, @main_menu.orderable
    assert_equal false, @main_menu.preorderable
    assert_equal "Änderungen erfolgreich gespeichert", flash[:notice]
    assert_redirected_to categories_url
  end

  test "category update not successfull with missing parameters" do
    @category = Category.new :name => "newCat", :rank => 1000, :orderable => true, :preorderable => true
    @category.save
    post :update, :id => @category.id, :category => {:name => "", :rank => '', :orderable => '', :preorderable => ''}

    #check if update did not affect the category
    @category = Category.find @category.id
    assert_equal @category.name, "newCat"
    assert_equal @category.rank, 1000
    assert_equal @category.orderable, true
    assert_equal @category.preorderable, true

    # Edit form should be shown to the user
    assert_template :edit
  end

  test "edit form is initialized with category object" do
    get :edit, :id => @main_menu.id

    assert_equal assigns(:category), @main_menu
    assert_template :edit
    assert_response :success
  end

  test "edit form not shown when id is not known" do
    get :edit

    assert_equal "Eine Kategorie mit dieser ID gibt es nicht!", flash[:error]
    assert_redirected_to categories_url
  end

  test "category is deleted by id" do
    @category = Category.new :name => "Category", :rank => 100, :orderable => true, :preorderable => true
    @category.save

    assert_difference 'Category.count', -1 do
      delete :destroy, :id => @category.id
    end

    assert_equal flash[:notice], "Kategorie #{@category.name} erfolgreich gelöscht!"
    assert_redirected_to categories_url
  end
end
